<?php
defined( 'ABSPATH' ) || exit;

class HRM_Leave {

    public static function init() {
        add_action( 'wp_ajax_hrm_apply_leave', [ __CLASS__, 'handle_ajax_apply_leave' ] );
        add_action( 'wp_ajax_nopriv_hrm_apply_leave', [ __CLASS__, 'handle_ajax_apply_leave' ] );
        add_action( 'wp_ajax_hrm_update_leave_status', [ __CLASS__, 'handle_ajax_update_leave_status' ] );
        add_action( 'wp_ajax_hrm_cancel_leave', [ __CLASS__, 'handle_ajax_cancel_leave' ] );
        add_action( 'wp_ajax_nopriv_hrm_cancel_leave', [ __CLASS__, 'handle_ajax_cancel_leave' ] );
        add_action( 'wp_ajax_hrm_edit_leave_request', [ __CLASS__, 'handle_ajax_edit_leave_request' ] );
        add_action( 'wp_ajax_nopriv_hrm_edit_leave_request', [ __CLASS__, 'handle_ajax_edit_leave_request' ] );
        add_action( 'wp_ajax_hrm_get_leave_data', [ __CLASS__, 'handle_ajax_get_leave_data' ] );
        add_action( 'wp_ajax_nopriv_hrm_get_leave_data', [ __CLASS__, 'handle_ajax_get_leave_data' ] );
        // Export actions
        add_action( 'wp_ajax_hrm_export_leave_csv', [ __CLASS__, 'handle_ajax_export_leave_csv' ] );
        add_action( 'wp_ajax_nopriv_hrm_export_leave_csv', [ __CLASS__, 'handle_ajax_export_leave_csv' ] );
        add_action( 'wp_ajax_hrm_export_leave_pdf', [ __CLASS__, 'handle_ajax_export_leave_pdf' ] );
        add_action( 'wp_ajax_nopriv_hrm_export_leave_pdf', [ __CLASS__, 'handle_ajax_export_leave_pdf' ] );
    }

    /**
     * Render Admin Leave Page
     */
    public static function render_admin_leave() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';

        // Get filter parameters
        $employee_filter = isset($_GET['employee']) ? intval($_GET['employee']) : 0;
        $type_filter = isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '';
        $status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        $date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '';
        $date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '';

        // Pagination
        $per_page = HRM_Pagination::get_per_page();
        $current_page = HRM_Pagination::get_current_page();
        $offset = HRM_Pagination::get_offset($current_page, $per_page);

        // Build query
        $where = ['1=1'];
        $params = [];

        if ($employee_filter) {
            $where[] = "employee_id = %d";
            $params[] = $employee_filter;
        }

        if ($type_filter) {
            $where[] = "leave_type = %s";
            $params[] = $type_filter;
        }

        if ($status_filter) {
            $where[] = "status = %s";
            $params[] = $status_filter;
        }

        if ($date_from) {
            $where[] = "start_date >= %s";
            $params[] = $date_from;
        }

        if ($date_to) {
            $where[] = "end_date <= %s";
            $params[] = $date_to;
        }

        $where_clause = implode(' AND ', $where);

        // Get total count
        if (!empty($params)) {
            $total_query = "SELECT COUNT(*) FROM $table_name WHERE $where_clause";
            $total_leaves = $wpdb->get_var($wpdb->prepare($total_query, $params));
        } else {
            $total_leaves = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where_clause");
        }

        // Get leaves
        if (!empty($params)) {
            $query = "SELECT * FROM $table_name WHERE $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d";
            $params[] = $per_page;
            $params[] = $offset;
            $leaves = $wpdb->get_results($wpdb->prepare($query, $params));
        } else {
            $query = "SELECT * FROM $table_name WHERE $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d";
            $leaves = $wpdb->get_results($wpdb->prepare($query, $per_page, $offset));
        }

        $total_pages = ceil($total_leaves / $per_page);

        // Get all employees for filter
        $employees = get_users(['role' => 'hrm_employee', 'orderby' => 'display_name']);

        // Get stats
        $stats = [
            'pending' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'pending'"),
            'approved' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'approved'"),
            'rejected' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE status = 'rejected'"),
            'total' => $wpdb->get_var("SELECT COUNT(*) FROM $table_name")
        ];

        ?>
        <div class="hrm-wrap">
            <!-- Header -->
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                    <h1>Leave Management</h1>
                    <p>Manage employee leave requests and approvals</p>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="hrm-stats-row">
                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Total Requests</span>
                        <span class="dashicons dashicons-calendar-alt"></span>
                    </div>
                    <div class="stat-number"><?php echo esc_html($stats['total']); ?></div>
                    <div class="stat-footer">All time requests</div>
                </div>

                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Pending</span>
                        <span class="dashicons dashicons-clock"></span>
                    </div>
                    <div class="stat-number" style="color: #F59E0B;"><?php echo esc_html($stats['pending']); ?></div>
                    <div class="stat-footer">Awaiting approval</div>
                </div>

                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Approved</span>
                        <span class="dashicons dashicons-yes-alt"></span>
                    </div>
                    <div class="stat-number" style="color: #10B981;"><?php echo esc_html($stats['approved']); ?></div>
                    <div class="stat-footer">Approved requests</div>
                </div>

                <div class="hrm-stat-card white">
                    <div class="stat-header">
                        <span>Rejected</span>
                        <span class="dashicons dashicons-dismiss"></span>
                    </div>
                    <div class="stat-number" style="color: #EF4444;"><?php echo esc_html($stats['rejected']); ?></div>
                    <div class="stat-footer">Rejected requests</div>
                </div>
            </div>

            <!-- Filters -->
            <div class="hrm-card">
                <div class="card-header">
                    <h3>Filter Leave Requests</h3>
                </div>
                <div class="hrm-filter-bar" style="padding: 4px 0;">
                    <form method="get" style="display: flex; gap: 10px; align-items: center; flex-wrap: nowrap; width: 100%;">
                        <input type="hidden" name="page" value="hrm-leave">
                        
                        <select name="employee" class="hrm-select-input" style="flex: 1 1 0; min-width: 0;">
                            <option value="">All Employees</option>
                            <?php foreach ($employees as $emp): ?>
                                <option value="<?php echo esc_attr($emp->ID); ?>" <?php selected($employee_filter, $emp->ID); ?>>
                                    <?php echo esc_html($emp->display_name); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>

                        <select name="type" class="hrm-select-input" style="flex: 1 1 0; min-width: 0;">
                            <option value="">All Types</option>
                            <option value="casual" <?php selected($type_filter, 'casual'); ?>>Casual</option>
                            <option value="sick" <?php selected($type_filter, 'sick'); ?>>Sick</option>
                            <option value="annual" <?php selected($type_filter, 'annual'); ?>>Annual</option>
                            <option value="unpaid" <?php selected($type_filter, 'unpaid'); ?>>Unpaid</option>
                        </select>

                        <select name="status" class="hrm-select-input" style="flex: 1 1 0; min-width: 0;">
                            <option value="">All Status</option>
                            <option value="pending" <?php selected($status_filter, 'pending'); ?>>Pending</option>
                            <option value="approved" <?php selected($status_filter, 'approved'); ?>>Approved</option>
                            <option value="rejected" <?php selected($status_filter, 'rejected'); ?>>Rejected</option>
                        </select>

                        <input type="date" name="date_from" class="hrm-date-input" placeholder="From" value="<?php echo esc_attr($date_from); ?>" style="flex: 1 1 0; min-width: 0;">
                        <input type="date" name="date_to" class="hrm-date-input" placeholder="To" value="<?php echo esc_attr($date_to); ?>" style="flex: 1 1 0; min-width: 0;">

                        <button type="submit" class="hrm-btn-primary" style="display: flex; align-items: center; gap: 6px; white-space: nowrap; flex-shrink: 0; padding: 10px 20px; font-size: 14px; line-height: 1.4;">
                            <span class="dashicons dashicons-search" style="font-size: 18px; line-height: 1; display: inline-block;"></span>
                            <span>Apply</span>
                        </button>
                    </form>
                </div>
            </div>

            <!-- Leave Requests Table -->
            <div class="hrm-card">
                <div class="card-header">
                    <h3>Leave Requests (<?php echo esc_html($total_leaves); ?>)</h3>
                </div>

                <?php if (empty($leaves)): ?>
                    <div style="padding: 60px 20px; text-align: center; color: #6B7280;">
                        <span class="dashicons dashicons-calendar-alt" style="font-size: 64px; opacity: 0.3;"></span>
                        <p style="margin-top: 16px; font-size: 16px;">No leave requests found</p>
                    </div>
                <?php else: ?>
                    <div class="hrm-table-container" style="overflow-x: auto;">
                        <table class="hrm-table hrm-leave-table">
                            <thead>
                                <tr>
                                    <th style="min-width: 150px;">Employee</th>
                                    <th style="min-width: 70px;">Type</th>
                                    <th style="min-width: 180px;">Dates</th>
                                    <th style="min-width: 50px; text-align: center;">Days</th>
                                    <th style="min-width: 150px;">Reason</th>
                                    <th style="min-width: 90px;">Status</th>
                                    <th style="min-width: 70px; text-align: center;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($leaves as $leave): 
                                    $employee = get_userdata($leave->employee_id);
                                    $employee_name = $employee ? $employee->display_name : 'Unknown';
                                    
                                    // Calculate days
                                    $start = new DateTime($leave->start_date);
                                    $end = new DateTime($leave->end_date);
                                    $days = $end->diff($start)->days + 1;
                                    
                                    // Status badge classes
                                    $status_classes = [
                                        'pending' => 'hrm-badge-warning',
                                        'approved' => 'hrm-badge-success',
                                        'rejected' => 'hrm-badge-danger'
                                    ];
                                    
                                    $badge_class = isset($status_classes[$leave->status]) ? $status_classes[$leave->status] : 'hrm-badge-default';
                                ?>
                                <tr class="hrm-leave-row" 
                                    data-leave-id="<?php echo esc_attr($leave->id); ?>"
                                    data-employee-name="<?php echo esc_attr($employee_name); ?>"
                                    data-leave-type="<?php echo esc_attr($leave->leave_type); ?>"
                                    data-start-date="<?php echo esc_attr(date('F d, Y', strtotime($leave->start_date))); ?>"
                                    data-end-date="<?php echo esc_attr(date('F d, Y', strtotime($leave->end_date))); ?>"
                                    data-days="<?php echo esc_attr($days); ?>"
                                    data-reason="<?php echo esc_attr($leave->reason); ?>"
                                    data-status="<?php echo esc_attr($leave->status); ?>"
                                    data-admin-remarks="<?php echo esc_attr($leave->admin_remarks); ?>"
                                    data-created="<?php echo esc_attr(date('M d, Y \a\t h:i A', strtotime($leave->created_at))); ?>"
                                    style="cursor: pointer;">
                                    <td>
                                        <div class="hrm-user-cell">
                                            <div class="hrm-avatar-sm"><?php echo esc_html(strtoupper(substr($employee_name, 0, 1))); ?></div>
                                            <span style="font-weight: 600; color: #111827;"><?php echo esc_html($employee_name); ?></span>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="hrm-leave-type-badge hrm-leave-type-<?php echo esc_attr($leave->leave_type); ?>">
                                            <?php echo esc_html(ucfirst($leave->leave_type)); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div style="white-space: nowrap;">
                                            <span style="color: #374151; font-weight: 500;"><?php echo esc_html(date('M d', strtotime($leave->start_date))); ?></span>
                                            <span style="color: #9CA3AF; margin: 0 6px;">→</span>
                                            <span style="color: #374151; font-weight: 500;"><?php echo esc_html(date('M d, Y', strtotime($leave->end_date))); ?></span>
                                        </div>
                                    </td>
                                    <td style="text-align: center;">
                                        <strong style="color: #374151; font-size: 14px;"><?php echo esc_html($days); ?></strong>
                                    </td>
                                    <td>
                                        <div class="hrm-text-truncate" title="<?php echo esc_attr($leave->reason); ?>">
                                            <?php echo esc_html($leave->reason); ?>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="hrm-badge <?php echo esc_attr($badge_class); ?>">
                                            <?php echo esc_html(strtoupper($leave->status)); ?>
                                        </span>
                                    </td>
                                    <td style="text-align: center;" class="hrm-actions-cell" onclick="event.stopPropagation();">
                                        <?php if ($leave->status === 'pending'): ?>
                                            <div style="display: inline-flex; gap: 6px;">
                                                <button class="hrm-action-btn hrm-action-approve hrm-approve-leave" 
                                                        data-leave-id="<?php echo esc_attr($leave->id); ?>"
                                                        title="Approve">
                                                    <span class="dashicons dashicons-yes-alt"></span>
                                                </button>
                                                <button class="hrm-action-btn hrm-action-reject hrm-reject-leave" 
                                                        data-leave-id="<?php echo esc_attr($leave->id); ?>"
                                                        title="Reject">
                                                    <span class="dashicons dashicons-dismiss"></span>
                                                </button>
                                            </div>
                                        <?php else: ?>
                                            <span style="color: #9CA3AF; font-size: 12px; white-space: nowrap;">
                                                <?php echo esc_html(date('M d', strtotime($leave->updated_at))); ?>
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    <?php 
                    HRM_Pagination::render([
                        'total_items' => $total_leaves,
                        'current_page' => $current_page,
                        'per_page' => $per_page,
                        'query_args' => [
                            'page' => 'hrm-leave',
                            'employee' => $employee_filter,
                            'type' => $type_filter,
                            'status' => $status_filter,
                            'date_from' => $date_from,
                            'date_to' => $date_to
                        ],
                        'show_sizes' => true,
                        'context' => 'admin'
                    ]);
                    HRM_Pagination::render_script();
                    ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Leave Details Modal -->
        <div id="hrm-leave-detail-modal" class="hrm-modal" style="display: none;">
            <div class="hrm-modal-content hrm-premium-modal" style="max-width: 580px;">
                <div class="hrm-modal-header-premium">
                    <div class="header-icon-wrapper" id="leave-modal-icon" style="background: rgba(12, 135, 118, 0.15); color: #0C8776;">
                        <span class="dashicons dashicons-calendar-alt"></span>
                    </div>
                    <div class="header-text">
                        <h2>Leave Request Details</h2>
                        <p id="leave-modal-subtitle">Submitted on <span id="leave-created-date">-</span></p>
                    </div>
                    <button class="hrm-modal-close-premium" onclick="closeLeaveModal()">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                
                <div class="hrm-modal-body-premium" style="padding: 24px;">
                    <!-- Status Banner -->
                    <div id="leave-status-banner" class="hrm-status-banner" style="margin-bottom: 24px; padding: 12px 16px; border-radius: 8px; display: flex; align-items: center; gap: 10px;">
                        <span class="dashicons" id="status-banner-icon"></span>
                        <span id="status-banner-text" style="font-weight: 600;"></span>
                    </div>
                    
                    <!-- Employee Info -->
                    <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 24px; padding-bottom: 20px; border-bottom: 1px solid #F3F4F6;">
                        <div class="hrm-avatar-lg" id="leave-emp-avatar" style="width: 50px; height: 50px; font-size: 20px; background: linear-gradient(135deg, #E0E7FF 0%, #C7D2FE 100%); color: #4338CA; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 700;">M</div>
                        <div>
                            <div id="leave-emp-name" style="font-size: 18px; font-weight: 700; color: #111827;">Employee Name</div>
                            <div style="display: flex; align-items: center; gap: 8px; margin-top: 4px;">
                                <span class="hrm-leave-type-badge" id="leave-type-badge">Casual</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Date Range -->
                    <div class="hrm-detail-grid" style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 16px; margin-bottom: 24px;">
                        <div class="hrm-detail-item">
                            <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">Start Date</label>
                            <div id="leave-start-date" style="font-size: 15px; font-weight: 600; color: #111827;">-</div>
                        </div>
                        <div class="hrm-detail-item">
                            <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">End Date</label>
                            <div id="leave-end-date" style="font-size: 15px; font-weight: 600; color: #111827;">-</div>
                        </div>
                        <div class="hrm-detail-item">
                            <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">Total Days</label>
                            <div id="leave-days" style="font-size: 24px; font-weight: 700; color: #0C8776;">0</div>
                        </div>
                    </div>
                    
                    <!-- Reason -->
                    <div style="margin-bottom: 24px;">
                        <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 8px;">Reason for Leave</label>
                        <div id="leave-reason" style="background: #F9FAFB; padding: 16px; border-radius: 8px; color: #374151; font-size: 14px; line-height: 1.6; border: 1px solid #E5E7EB;">
                            -
                        </div>
                    </div>
                    
                    <!-- Admin Remarks (shown if exists) -->
                    <div id="admin-remarks-section" style="margin-bottom: 24px; display: none;">
                        <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 8px;">Admin Remarks</label>
                        <div id="leave-admin-remarks" style="background: #FEF3C7; padding: 16px; border-radius: 8px; color: #92400E; font-size: 14px; line-height: 1.6; border: 1px solid #FCD34D;">
                            -
                        </div>
                    </div>
                    
                    <!-- Action Section (only for pending) -->
                    <div id="leave-action-section" style="display: none; padding-top: 20px; border-top: 1px solid #F3F4F6;">
                        <form id="hrm-leave-detail-form">
                            <input type="hidden" id="detail-leave-id" name="leave_id">
                            
                            <div class="hrm-form-group" style="margin-bottom: 20px;">
                                <label style="display: block; font-size: 13px; font-weight: 600; color: #374151; margin-bottom: 8px;">Admin Remarks <span style="color: #9CA3AF; font-weight: 400;">(optional)</span></label>
                                <textarea id="detail-admin-remarks" name="admin_remarks" class="hrm-input" rows="3" 
                                          placeholder="Add any comments or notes about this decision..."
                                          style="width: 100%; padding: 12px; border: 1px solid #E5E7EB; border-radius: 8px; resize: vertical;"></textarea>
                            </div>
                            
                            <div style="display: flex; gap: 12px; justify-content: flex-end;">
                                <button type="button" class="hrm-btn-outline" onclick="closeLeaveModal()">Cancel</button>
                                <button type="button" class="hrm-btn-danger" id="btn-reject-leave" style="background: #FEE2E2; color: #DC2626; border: none;">
                                    <span class="dashicons dashicons-dismiss" style="margin-right: 4px;"></span> Reject
                                </button>
                                <button type="button" class="hrm-btn-primary" id="btn-approve-leave" style="background: #10B981;">
                                    <span class="dashicons dashicons-yes-alt" style="margin-right: 4px;"></span> Approve
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            
            // Close leave modal function
            window.closeLeaveModal = function() {
                $('#hrm-leave-detail-modal').fadeOut(200);
                $('#hrm-leave-detail-form')[0].reset();
                $('body').removeClass('hrm-modal-open');
            };
            
            // Open leave detail modal on row click
            $('.hrm-leave-row').on('click', function(e) {
                // Don't open if clicking on action buttons
                if ($(e.target).closest('.hrm-actions-cell').length) {
                    return;
                }
                
                const row = $(this);
                const leaveId = row.data('leave-id');
                const employeeName = row.data('employee-name');
                const leaveType = row.data('leave-type');
                const startDate = row.data('start-date');
                const endDate = row.data('end-date');
                const days = row.data('days');
                const reason = row.data('reason');
                const status = row.data('status');
                const adminRemarks = row.data('admin-remarks');
                const created = row.data('created');
                
                // Populate modal
                $('#leave-emp-avatar').text(employeeName.charAt(0).toUpperCase());
                $('#leave-emp-name').text(employeeName);
                $('#leave-type-badge').text(leaveType.charAt(0).toUpperCase() + leaveType.slice(1))
                    .removeClass('hrm-leave-type-casual hrm-leave-type-sick hrm-leave-type-annual hrm-leave-type-unpaid')
                    .addClass('hrm-leave-type-' + leaveType);
                $('#leave-start-date').text(startDate);
                $('#leave-end-date').text(endDate);
                $('#leave-days').text(days);
                $('#leave-reason').text(reason || 'No reason provided');
                $('#leave-created-date').text(created);
                $('#detail-leave-id').val(leaveId);
                
                // Status banner styling
                const statusBanner = $('#leave-status-banner');
                const statusIcon = $('#status-banner-icon');
                const statusText = $('#status-banner-text');
                
                if (status === 'pending') {
                    statusBanner.css({'background': '#FEF3C7', 'color': '#92400E'});
                    statusIcon.attr('class', 'dashicons dashicons-clock');
                    statusText.text('Pending Approval');
                    $('#leave-action-section').show();
                    $('#admin-remarks-section').hide();
                } else if (status === 'approved') {
                    statusBanner.css({'background': '#D1FAE5', 'color': '#065F46'});
                    statusIcon.attr('class', 'dashicons dashicons-yes-alt');
                    statusText.text('Approved');
                    $('#leave-action-section').hide();
                } else if (status === 'rejected') {
                    statusBanner.css({'background': '#FEE2E2', 'color': '#991B1B'});
                    statusIcon.attr('class', 'dashicons dashicons-dismiss');
                    statusText.text('Rejected');
                    $('#leave-action-section').hide();
                }
                
                // Show admin remarks if exists
                if (adminRemarks && adminRemarks.trim() !== '') {
                    $('#leave-admin-remarks').text(adminRemarks);
                    $('#admin-remarks-section').show();
                } else {
                    $('#admin-remarks-section').hide();
                }
                
                // Show modal - add body class for proper isolation
                $('body').addClass('hrm-modal-open');
                $('#hrm-leave-detail-modal').fadeIn(200);
            });
            
            // Handle approve button
            $('#btn-approve-leave').on('click', function() {
                submitLeaveAction('approved');
            });
            
            // Handle reject button
            $('#btn-reject-leave').on('click', function() {
                submitLeaveAction('rejected');
            });
            
            // Submit leave action
            function submitLeaveAction(status) {
                const leaveId = $('#detail-leave-id').val();
                const remarks = $('#detail-admin-remarks').val();
                
                const approveBtn = $('#btn-approve-leave');
                const rejectBtn = $('#btn-reject-leave');
                
                approveBtn.prop('disabled', true);
                rejectBtn.prop('disabled', true);
                
                const activeBtn = status === 'approved' ? approveBtn : rejectBtn;
                activeBtn.html('<span class="dashicons dashicons-update hrm-spin"></span> Processing...');
                
                $.ajax({
                    url: hrm_admin_params.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_update_leave_status',
                        nonce: hrm_admin_params.nonce,
                        leave_id: leaveId,
                        status: status,
                        admin_remarks: remarks
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || 'Error updating leave status');
                            resetButtons();
                        }
                    },
                    error: function() {
                        alert('Server error. Please try again.');
                        resetButtons();
                    }
                });
                
                function resetButtons() {
                    approveBtn.prop('disabled', false).html('<span class="dashicons dashicons-yes-alt" style="margin-right: 4px;"></span> Approve');
                    rejectBtn.prop('disabled', false).html('<span class="dashicons dashicons-dismiss" style="margin-right: 4px;"></span> Reject');
                }
            }
            
            // Quick approve/reject from table buttons
            $('.hrm-approve-leave').on('click', function(e) {
                e.stopPropagation();
                const row = $(this).closest('.hrm-leave-row');
                row.trigger('click');
            });
            
            $('.hrm-reject-leave').on('click', function(e) {
                e.stopPropagation();
                const row = $(this).closest('.hrm-leave-row');
                row.trigger('click');
            });
            
            // Close modal on overlay click
            $('#hrm-leave-detail-modal').on('click', function(e) {
                if ($(e.target).is('.hrm-modal')) {
                    closeLeaveModal();
                }
            });
            
            // Close on escape key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('#hrm-leave-detail-modal').is(':visible')) {
                    closeLeaveModal();
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Render Employee Leave Interface (Frontend)
     */
    public static function render_employee_leave() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        // Get employee's leave history
        $leaves = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE employee_id = %d ORDER BY created_at DESC",
            $current_user->ID
        ));

        // Get stats
        $stats = [
            'pending' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE employee_id = %d AND status = 'pending'",
                $current_user->ID
            )),
            'approved' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE employee_id = %d AND status = 'approved'",
                $current_user->ID
            )),
            'rejected' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE employee_id = %d AND status = 'rejected'",
                $current_user->ID
            ))
        ];

        ?>
        <div class="hrm-leave-section">
            <div class="hrm-section-header">
                <h2><span class="dashicons dashicons-calendar-alt"></span> Leave Management</h2>
                <button class="hrm-btn-primary" id="hrm-apply-leave-btn">
                    <span class="dashicons dashicons-plus"></span> Apply for Leave
                </button>
            </div>

            <!-- Stats -->
            <div class="hrm-leave-stats">
                <div class="hrm-leave-stat-card" style="background: linear-gradient(135deg, #F59E0B 0%, #D97706 100%);">
                    <div class="stat-icon"><span class="dashicons dashicons-clock"></span></div>
                    <div class="stat-value"><?php echo esc_html($stats['pending']); ?></div>
                    <div class="stat-label">Pending</div>
                </div>
                <div class="hrm-leave-stat-card" style="background: linear-gradient(135deg, #10B981 0%, #059669 100%);">
                    <div class="stat-icon"><span class="dashicons dashicons-yes-alt"></span></div>
                    <div class="stat-value"><?php echo esc_html($stats['approved']); ?></div>
                    <div class="stat-label">Approved</div>
                </div>
                <div class="hrm-leave-stat-card" style="background: linear-gradient(135deg, #EF4444 0%, #DC2626 100%);">
                    <div class="stat-icon"><span class="dashicons dashicons-dismiss"></span></div>
                    <div class="stat-value"><?php echo esc_html($stats['rejected']); ?></div>
                    <div class="stat-label">Rejected</div>
                </div>
            </div>

            <!-- Leave History -->
            <div class="hrm-leave-history-section">
                <div class="hrm-section-title">
                    <h3><span class="dashicons dashicons-list-view"></span> Leave History</h3>
                    <div class="hrm-history-controls">
                        <div class="hrm-page-size-control">
                            <label for="hrm-leave-page-size">Show:</label>
                            <select id="hrm-leave-page-size" class="hrm-select-sm">
                                <option value="3" selected>3</option>
                                <option value="10">10</option>
                                <option value="20">20</option>
                                <option value="50">50</option>
                            </select>
                            <span class="hrm-page-size-label">per page</span>
                        </div>
                        <?php if (!empty($leaves)): ?>
                        <div class="hrm-export-controls">
                            <div class="hrm-export-dropdown">
                                <button type="button" class="hrm-btn-export" id="hrm-export-btn">
                                    <span class="dashicons dashicons-download"></span> Export
                                    <span class="dashicons dashicons-arrow-down-alt2"></span>
                                </button>
                                <div class="hrm-export-menu" id="hrm-export-menu">
                                    <button type="button" class="hrm-export-option" id="hrm-export-csv">
                                        <span class="dashicons dashicons-media-spreadsheet"></span> Export as CSV
                                    </button>
                                    <button type="button" class="hrm-export-option" id="hrm-export-pdf">
                                        <span class="dashicons dashicons-pdf"></span> Export as PDF
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                        <span class="hrm-badge-count"><?php echo count($leaves); ?> request(s)</span>
                    </div>
                </div>
                
                <?php if (empty($leaves)): ?>
                    <div class="hrm-empty-state-modern">
                        <div class="empty-icon">
                            <span class="dashicons dashicons-calendar-alt"></span>
                        </div>
                        <h4>No Leave Requests Yet</h4>
                        <p>You haven't applied for any leave. Click the button below to submit your first leave request.</p>
                        <button class="hrm-btn-primary" onclick="document.getElementById('hrm-apply-leave-btn').click()">
                            <span class="dashicons dashicons-plus"></span> Apply for Leave
                        </button>
                    </div>
                <?php else: ?>
                    <div class="hrm-leave-cards" id="hrm-leave-cards-container">
                        <?php $card_index = 0; foreach ($leaves as $leave): 
                            $start = new DateTime($leave->start_date);
                            $end = new DateTime($leave->end_date);
                            $days = $end->diff($start)->days + 1;
                            
                            // Status classes
                            $status_colors = [
                                'pending' => ['bg' => '#FEF3C7', 'border' => '#F59E0B', 'text' => '#92400E', 'icon' => 'dashicons-clock'],
                                'approved' => ['bg' => '#D1FAE5', 'border' => '#10B981', 'text' => '#065F46', 'icon' => 'dashicons-yes-alt'],
                                'rejected' => ['bg' => '#FEE2E2', 'border' => '#EF4444', 'text' => '#991B1B', 'icon' => 'dashicons-dismiss']
                            ];
                            $status_style = $status_colors[$leave->status] ?? $status_colors['pending'];
                        ?>
                        <div class="hrm-leave-card hrm-leave-card-clickable hrm-leave-card-paginated" 
                             data-card-index="<?php echo $card_index; ?>"
                             data-leave-id="<?php echo esc_attr($leave->id); ?>"
                             data-leave-type="<?php echo esc_attr($leave->leave_type); ?>"
                             data-start-date="<?php echo esc_attr(date('F d, Y', strtotime($leave->start_date))); ?>"
                             data-end-date="<?php echo esc_attr(date('F d, Y', strtotime($leave->end_date))); ?>"
                             data-days="<?php echo esc_attr($days); ?>"
                             data-reason="<?php echo esc_attr($leave->reason); ?>"
                             data-status="<?php echo esc_attr($leave->status); ?>"
                             data-admin-remarks="<?php echo esc_attr($leave->admin_remarks); ?>"
                             data-created="<?php echo esc_attr(date('M d, Y \a\t h:i A', strtotime($leave->created_at))); ?>"
                             style="border-left: 4px solid <?php echo $status_style['border']; ?>; cursor: pointer;">
                            <!-- Card Header -->
                            <div class="hrm-leave-card-header">
                                <div class="hrm-leave-card-left">
                                    <span class="hrm-leave-type-badge hrm-leave-type-<?php echo esc_attr($leave->leave_type); ?>">
                                        <?php echo esc_html(ucfirst($leave->leave_type)); ?>
                                    </span>
                                    <span class="hrm-leave-submitted">
                                        Applied <?php echo esc_html(human_time_diff(strtotime($leave->created_at), current_time('timestamp'))); ?> ago
                                    </span>
                                </div>
                                <div class="hrm-leave-card-right">
                                    <span class="hrm-status-pill" style="background: <?php echo $status_style['bg']; ?>; color: <?php echo $status_style['text']; ?>;">
                                        <span class="dashicons <?php echo $status_style['icon']; ?>"></span>
                                        <?php echo esc_html(ucfirst($leave->status)); ?>
                                    </span>
                                </div>
                            </div>
                            
                            <!-- Card Body -->
                            <div class="hrm-leave-card-body">
                                <!-- Date Range -->
                                <div class="hrm-leave-date-range">
                                    <div class="hrm-date-block">
                                        <span class="hrm-date-label">From</span>
                                        <span class="hrm-date-value"><?php echo esc_html(date('M d', strtotime($leave->start_date))); ?></span>
                                        <span class="hrm-date-year"><?php echo esc_html(date('Y', strtotime($leave->start_date))); ?></span>
                                    </div>
                                    <div class="hrm-date-arrow">
                                        <span class="dashicons dashicons-arrow-right-alt"></span>
                                    </div>
                                    <div class="hrm-date-block">
                                        <span class="hrm-date-label">To</span>
                                        <span class="hrm-date-value"><?php echo esc_html(date('M d', strtotime($leave->end_date))); ?></span>
                                        <span class="hrm-date-year"><?php echo esc_html(date('Y', strtotime($leave->end_date))); ?></span>
                                    </div>
                                    <div class="hrm-days-count">
                                        <span class="days-number"><?php echo esc_html($days); ?></span>
                                        <span class="days-label">Day<?php echo $days > 1 ? 's' : ''; ?></span>
                                    </div>
                                </div>
                                
                                <!-- Reason -->
                                <div class="hrm-leave-reason-box">
                                    <span class="reason-label">Reason:</span>
                                    <span class="reason-text"><?php echo esc_html($leave->reason); ?></span>
                                </div>
                                
                                <!-- Admin Remarks (if any) -->
                                <?php if ($leave->admin_remarks): ?>
                                <div class="hrm-admin-remarks-box" style="background: <?php echo $status_style['bg']; ?>;">
                                    <span class="dashicons dashicons-admin-comments"></span>
                                    <div>
                                        <span class="remarks-label">Admin Remarks:</span>
                                        <span class="remarks-text"><?php echo esc_html($leave->admin_remarks); ?></span>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Card Footer (only for pending) -->
                            <?php if ($leave->status === 'pending'): ?>
                            <div class="hrm-leave-card-footer" onclick="event.stopPropagation();">
                                <button class="hrm-btn-sm hrm-btn-outline-primary hrm-edit-leave" data-leave-id="<?php echo esc_attr($leave->id); ?>">
                                    <span class="dashicons dashicons-edit"></span> Edit
                                </button>
                                <button class="hrm-btn-sm hrm-btn-outline-danger hrm-cancel-leave" data-leave-id="<?php echo esc_attr($leave->id); ?>">
                                    <span class="dashicons dashicons-no-alt"></span> Cancel Request
                                </button>
                            </div>
                            <?php endif; ?>
                        </div>
                        <?php $card_index++; endforeach; ?>
                    </div>
                    
                    <!-- Pagination Controls -->
                    <?php if (count($leaves) > 3): ?>
                    <div class="hrm-leave-pagination" id="hrm-leave-pagination">
                        <div class="hrm-pagination-info">
                            <span id="hrm-pagination-showing">Showing 1-3 of <?php echo count($leaves); ?></span>
                        </div>
                        <div class="hrm-pagination-controls">
                            <button type="button" class="hrm-pagination-btn" id="hrm-prev-page" disabled>
                                <span class="dashicons dashicons-arrow-left-alt2"></span> Previous
                            </button>
                            <div class="hrm-page-numbers" id="hrm-page-numbers"></div>
                            <button type="button" class="hrm-pagination-btn" id="hrm-next-page">
                                Next <span class="dashicons dashicons-arrow-right-alt2"></span>
                            </button>
                        </div>
                    </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Apply/Edit Leave Modal -->
        <div id="hrm-leave-modal" class="hrm-modal" style="display: none;">
            <div class="hrm-modal-content">
                <div class="hrm-modal-header">
                    <h2 id="hrm-leave-modal-title">Apply for Leave</h2>
                    <span class="hrm-modal-close">&times;</span>
                </div>
                <div class="hrm-modal-body">
                    <form id="hrm-leave-form">
                        <input type="hidden" id="hrm-leave-id" name="leave_id" value="">
                        
                        <div class="hrm-form-group">
                            <label for="hrm-leave-type">Leave Type <span class="required">*</span></label>
                            <select id="hrm-leave-type" name="leave_type" class="hrm-input" required>
                                <option value="">Select Leave Type</option>
                                <option value="casual">Casual Leave</option>
                                <option value="sick">Sick Leave</option>
                                <option value="annual">Annual Leave</option>
                                <option value="unpaid">Unpaid Leave</option>
                            </select>
                        </div>

                        <div class="hrm-form-row">
                            <div class="hrm-form-group">
                                <label for="hrm-start-date">Start Date <span class="required">*</span></label>
                                <input type="date" id="hrm-start-date" name="start_date" class="hrm-input" required>
                            </div>

                            <div class="hrm-form-group">
                                <label for="hrm-end-date">End Date <span class="required">*</span></label>
                                <input type="date" id="hrm-end-date" name="end_date" class="hrm-input" required>
                            </div>
                        </div>

                        <div class="hrm-form-group">
                            <label for="hrm-leave-reason">Reason <span class="required">*</span></label>
                            <textarea id="hrm-leave-reason" name="reason" class="hrm-input" rows="4" 
                                      placeholder="Please provide a detailed reason for your leave request" required></textarea>
                        </div>

                        <div class="hrm-form-actions">
                            <button type="button" class="hrm-btn-outline hrm-modal-close">Cancel</button>
                            <button type="submit" class="hrm-btn-primary">
                                <span class="dashicons dashicons-yes"></span> Submit Request
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Leave Detail View Modal (Employee) -->
        <div id="hrm-leave-view-modal" class="hrm-modal" style="display: none;">
            <div class="hrm-modal-content" style="max-width: 520px;">
                <div class="hrm-modal-header" style="background: linear-gradient(135deg, #064E4C 0%, #0C8776 100%); color: white; border-radius: 12px 12px 0 0; padding: 20px 24px;">
                    <div style="display: flex; align-items: center; gap: 12px;">
                        <div style="width: 40px; height: 40px; background: rgba(255,255,255,0.2); border-radius: 10px; display: flex; align-items: center; justify-content: center;">
                            <span class="dashicons dashicons-calendar-alt" style="font-size: 20px;"></span>
                        </div>
                        <div>
                            <h2 style="margin: 0; font-size: 18px; font-weight: 700;">Leave Request Details</h2>
                            <p id="view-leave-submitted" style="margin: 4px 0 0 0; font-size: 13px; opacity: 0.9;">Submitted on -</p>
                        </div>
                    </div>
                    <span class="hrm-modal-close" style="color: white; opacity: 0.8;" onclick="closeViewModal()">&times;</span>
                </div>
                <div class="hrm-modal-body" style="padding: 24px;">
                    <!-- Status Banner -->
                    <div id="view-status-banner" style="padding: 12px 16px; border-radius: 10px; display: flex; align-items: center; gap: 10px; margin-bottom: 24px;">
                        <span class="dashicons" id="view-status-icon"></span>
                        <span id="view-status-text" style="font-weight: 600; font-size: 14px;"></span>
                    </div>

                    <!-- Leave Type -->
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">Leave Type</label>
                        <span class="hrm-leave-type-badge" id="view-leave-type" style="font-size: 14px; padding: 6px 14px;">-</span>
                    </div>

                    <!-- Date Range -->
                    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 16px; margin-bottom: 24px; background: #F9FAFB; padding: 16px; border-radius: 10px;">
                        <div style="text-align: center;">
                            <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">Start Date</label>
                            <div id="view-start-date" style="font-size: 15px; font-weight: 600; color: #111827;">-</div>
                        </div>
                        <div style="text-align: center;">
                            <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">End Date</label>
                            <div id="view-end-date" style="font-size: 15px; font-weight: 600; color: #111827;">-</div>
                        </div>
                        <div style="text-align: center;">
                            <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 6px;">Total Days</label>
                            <div id="view-days" style="font-size: 24px; font-weight: 800; color: #0C8776;">0</div>
                        </div>
                    </div>

                    <!-- Reason -->
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 8px;">Reason for Leave</label>
                        <div id="view-reason" style="background: #F9FAFB; padding: 16px; border-radius: 8px; color: #374151; font-size: 14px; line-height: 1.6; border: 1px solid #E5E7EB;">
                            -
                        </div>
                    </div>

                    <!-- Admin Remarks (if any) -->
                    <div id="view-remarks-section" style="display: none; margin-bottom: 20px;">
                        <label style="display: block; font-size: 11px; font-weight: 600; color: #9CA3AF; text-transform: uppercase; margin-bottom: 8px;">Admin Remarks</label>
                        <div id="view-admin-remarks" style="padding: 16px; border-radius: 8px; font-size: 14px; line-height: 1.6;">
                            -
                        </div>
                    </div>

                    <!-- Close Button -->
                    <div style="text-align: center; padding-top: 16px; border-top: 1px solid #F3F4F6;">
                        <button type="button" class="hrm-btn-primary" onclick="closeViewModal()" style="padding: 10px 32px;">
                            <span class="dashicons dashicons-yes"></span> Close
                        </button>
                    </div>
                </div>
            </div>
        </div>
        <script>
        jQuery(document).ready(function($) {
            // Open apply leave modal
            $('#hrm-apply-leave-btn').on('click', function() {
                $('#hrm-leave-modal-title').text('Apply for Leave');
                $('#hrm-leave-form')[0].reset();
                $('#hrm-leave-id').val('');
                $('body').addClass('hrm-modal-open');
                $('#hrm-leave-modal').fadeIn(200);
            });

            // Edit leave
            $('.hrm-edit-leave').on('click', function() {
                const leaveId = $(this).data('leave-id');
                
                $.ajax({
                    url: hrm_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_get_leave_data',
                        nonce: hrm_ajax.nonce,
                        leave_id: leaveId
                    },
                    success: function(response) {
                        if (response.success) {
                            const leave = response.data;
                            $('#hrm-leave-modal-title').text('Edit Leave Request');
                            $('#hrm-leave-id').val(leave.id);
                            $('#hrm-leave-type').val(leave.leave_type);
                            $('#hrm-start-date').val(leave.start_date);
                            $('#hrm-end-date').val(leave.end_date);
                            $('#hrm-leave-reason').val(leave.reason);
                            $('body').addClass('hrm-modal-open');
                            $('#hrm-leave-modal').fadeIn(200);
                        } else {
                            alert('Error loading leave data');
                        }
                    }
                });
            });

            // Cancel leave
            $('.hrm-cancel-leave').on('click', function() {
                if (!confirm('Are you sure you want to cancel this leave request?')) {
                    return;
                }

                const leaveId = $(this).data('leave-id');
                const btn = $(this);
                btn.addClass('loading').prop('disabled', true);

                $.ajax({
                    url: hrm_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_cancel_leave',
                        nonce: hrm_ajax.nonce,
                        leave_id: leaveId
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || 'Error cancelling leave');
                            btn.removeClass('loading').prop('disabled', false);
                        }
                    }
                });
            });

            // Close modal
            $('.hrm-modal-close').on('click', function() {
                $('#hrm-leave-modal').fadeOut(200);
                $('body').removeClass('hrm-modal-open');
            });

            // Submit leave form
            $('#hrm-leave-form').on('submit', function(e) {
                e.preventDefault();

                const submitBtn = $(this).find('button[type="submit"]');
                submitBtn.addClass('loading').prop('disabled', true);

                const leaveId = $('#hrm-leave-id').val();
                const action = leaveId ? 'hrm_edit_leave_request' : 'hrm_apply_leave';

                $.ajax({
                    url: hrm_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: action,
                        nonce: hrm_ajax.nonce,
                        leave_id: leaveId,
                        leave_type: $('#hrm-leave-type').val(),
                        start_date: $('#hrm-start-date').val(),
                        end_date: $('#hrm-end-date').val(),
                        reason: $('#hrm-leave-reason').val()
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || 'Error submitting leave request');
                            submitBtn.removeClass('loading').prop('disabled', false);
                        }
                    },
                    error: function() {
                        alert('Server error. Please try again.');
                        submitBtn.removeClass('loading').prop('disabled', false);
                    }
                });
            });

            // Set minimum date to today
            const today = new Date().toISOString().split('T')[0];
            $('#hrm-start-date, #hrm-end-date').attr('min', today);

            // Validate end date
            $('#hrm-start-date').on('change', function() {
                $('#hrm-end-date').attr('min', $(this).val());
            });
            
            // ==========================================
            // Leave Card Click - View Details Modal
            // ==========================================
            
            // Close view modal function
            window.closeViewModal = function() {
                $('#hrm-leave-view-modal').fadeOut(200);
                $('body').removeClass('hrm-modal-open');
            };
            
            // Open leave detail modal on card click
            $('.hrm-leave-card-clickable').on('click', function(e) {
                // Don't open if clicking on footer buttons
                if ($(e.target).closest('.hrm-leave-card-footer').length) {
                    return;
                }
                
                const card = $(this);
                const leaveType = card.data('leave-type');
                const startDate = card.data('start-date');
                const endDate = card.data('end-date');
                const days = card.data('days');
                const reason = card.data('reason');
                const status = card.data('status');
                const adminRemarks = card.data('admin-remarks');
                const created = card.data('created');
                
                // Populate modal
                $('#view-leave-submitted').text('Submitted on ' + created);
                $('#view-leave-type')
                    .text(leaveType.charAt(0).toUpperCase() + leaveType.slice(1))
                    .removeClass('hrm-leave-type-casual hrm-leave-type-sick hrm-leave-type-annual hrm-leave-type-unpaid')
                    .addClass('hrm-leave-type-' + leaveType);
                $('#view-start-date').text(startDate);
                $('#view-end-date').text(endDate);
                $('#view-days').text(days);
                $('#view-reason').text(reason || 'No reason provided');
                
                // Status banner styling
                const statusBanner = $('#view-status-banner');
                const statusIcon = $('#view-status-icon');
                const statusText = $('#view-status-text');
                const remarksSection = $('#view-remarks-section');
                const remarksBox = $('#view-admin-remarks');
                
                if (status === 'pending') {
                    statusBanner.css({'background': '#FEF3C7', 'color': '#92400E'});
                    statusIcon.attr('class', 'dashicons dashicons-clock');
                    statusText.text('Pending Approval');
                } else if (status === 'approved') {
                    statusBanner.css({'background': '#D1FAE5', 'color': '#065F46'});
                    statusIcon.attr('class', 'dashicons dashicons-yes-alt');
                    statusText.text('Approved');
                } else if (status === 'rejected') {
                    statusBanner.css({'background': '#FEE2E2', 'color': '#991B1B'});
                    statusIcon.attr('class', 'dashicons dashicons-dismiss');
                    statusText.text('Rejected');
                }
                
                // Show admin remarks if exists
                if (adminRemarks && adminRemarks.trim() !== '') {
                    remarksBox.text(adminRemarks);
                    if (status === 'approved') {
                        remarksBox.css({'background': '#D1FAE5', 'color': '#065F46', 'border': '1px solid #A7F3D0'});
                    } else if (status === 'rejected') {
                        remarksBox.css({'background': '#FEE2E2', 'color': '#991B1B', 'border': '1px solid #FECACA'});
                    } else {
                        remarksBox.css({'background': '#FEF3C7', 'color': '#92400E', 'border': '1px solid #FCD34D'});
                    }
                    remarksSection.show();
                } else {
                    remarksSection.hide();
                }
                
                // Show modal
                $('body').addClass('hrm-modal-open');
                $('#hrm-leave-view-modal').fadeIn(200);
            });
            
            // Close view modal on overlay click
            $('#hrm-leave-view-modal').on('click', function(e) {
                if ($(e.target).is('.hrm-modal')) {
                    closeViewModal();
                }
            });
            
            // Close on escape key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    if ($('#hrm-leave-view-modal').is(':visible')) {
                        closeViewModal();
                    }
                    if ($('#hrm-leave-modal').is(':visible')) {
                        $('#hrm-leave-modal').fadeOut(200);
                    }
                }
            });
            
            // ==========================================
            // Leave History Pagination
            // ==========================================
            
            const leaveCards = $('.hrm-leave-card-paginated');
            const totalCards = leaveCards.length;
            let currentPage = 1;
            let pageSize = parseInt($('#hrm-leave-page-size').val()) || 3;
            
            // Initialize pagination
            function initPagination() {
                if (totalCards <= 0) return;
                
                updatePagination();
                showCurrentPage();
            }
            
            // Calculate total pages
            function getTotalPages() {
                return Math.ceil(totalCards / pageSize);
            }
            
            // Show cards for current page
            function showCurrentPage() {
                const startIndex = (currentPage - 1) * pageSize;
                const endIndex = startIndex + pageSize;
                
                leaveCards.each(function(index) {
                    const card = $(this);
                    if (index >= startIndex && index < endIndex) {
                        card.removeClass('hrm-hidden').css('display', '');
                    } else {
                        card.addClass('hrm-hidden').css('display', 'none');
                    }
                });
                
                // Update pagination info
                const showingStart = startIndex + 1;
                const showingEnd = Math.min(endIndex, totalCards);
                $('#hrm-pagination-showing').text(`Showing ${showingStart}-${showingEnd} of ${totalCards}`);
            }
            
            // Update pagination controls
            function updatePagination() {
                const totalPages = getTotalPages();
                
                // Show/hide pagination if needed
                if (totalCards <= pageSize) {
                    $('#hrm-leave-pagination').hide();
                } else {
                    $('#hrm-leave-pagination').show();
                }
                
                // Update prev/next buttons
                $('#hrm-prev-page').prop('disabled', currentPage === 1);
                $('#hrm-next-page').prop('disabled', currentPage >= totalPages);
                
                // Generate page numbers
                const pageNumbers = $('#hrm-page-numbers');
                pageNumbers.empty();
                
                // Show max 5 page numbers
                let startPage = Math.max(1, currentPage - 2);
                let endPage = Math.min(totalPages, startPage + 4);
                
                if (endPage - startPage < 4) {
                    startPage = Math.max(1, endPage - 4);
                }
                
                for (let i = startPage; i <= endPage; i++) {
                    const activeClass = i === currentPage ? 'active' : '';
                    pageNumbers.append(`<button type="button" class="hrm-page-num ${activeClass}" data-page="${i}">${i}</button>`);
                }
                
                // Bind click events to page numbers
                pageNumbers.find('.hrm-page-num').on('click', function() {
                    currentPage = parseInt($(this).data('page'));
                    showCurrentPage();
                    updatePagination();
                });
            }
            
            // Page size change handler
            $('#hrm-leave-page-size').on('change', function() {
                pageSize = parseInt($(this).val());
                currentPage = 1; // Reset to first page
                showCurrentPage();
                updatePagination();
            });
            
            // Previous page button
            $('#hrm-prev-page').on('click', function() {
                if (currentPage > 1) {
                    currentPage--;
                    showCurrentPage();
                    updatePagination();
                }
            });
            
            // Next page button
            $('#hrm-next-page').on('click', function() {
                if (currentPage < getTotalPages()) {
                    currentPage++;
                    showCurrentPage();
                    updatePagination();
                }
            });
            
            // Initialize pagination on page load
            if (totalCards > 0) {
                initPagination();
            }
            
            // ==========================================
            // Export Functionality
            // ==========================================
            
            // Toggle export dropdown
            $('#hrm-export-btn').on('click', function(e) {
                e.stopPropagation();
                $('#hrm-export-menu').toggleClass('active');
            });
            
            // Close dropdown when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.hrm-export-dropdown').length) {
                    $('#hrm-export-menu').removeClass('active');
                }
            });
            
            // Export as CSV
            $('#hrm-export-csv').on('click', function() {
                const btn = $(this);
                btn.prop('disabled', true).html('<span class="dashicons dashicons-update hrm-spin"></span> Exporting...');
                
                $.ajax({
                    url: hrm_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_export_leave_csv',
                        nonce: hrm_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            // Create and download CSV file
                            const blob = new Blob([response.data.content], { type: 'text/csv;charset=utf-8;' });
                            const link = document.createElement('a');
                            link.href = URL.createObjectURL(blob);
                            link.download = response.data.filename;
                            link.click();
                            URL.revokeObjectURL(link.href);
                            
                            $('#hrm-export-menu').removeClass('active');
                        } else {
                            alert(response.data.message || 'Export failed');
                        }
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-media-spreadsheet"></span> Export as CSV');
                    },
                    error: function() {
                        alert('Server error. Please try again.');
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-media-spreadsheet"></span> Export as CSV');
                    }
                });
            });
            
            // Export as PDF
            $('#hrm-export-pdf').on('click', function() {
                const btn = $(this);
                btn.prop('disabled', true).html('<span class="dashicons dashicons-update hrm-spin"></span> Generating...');
                
                $.ajax({
                    url: hrm_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'hrm_export_leave_pdf',
                        nonce: hrm_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            generatePDF(response.data);
                            $('#hrm-export-menu').removeClass('active');
                        } else {
                            alert(response.data.message || 'Export failed');
                        }
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-pdf"></span> Export as PDF');
                    },
                    error: function() {
                        alert('Server error. Please try again.');
                        btn.prop('disabled', false).html('<span class="dashicons dashicons-pdf"></span> Export as PDF');
                    }
                });
            });
            
            // Generate PDF from data using jsPDF
            function generatePDF(data) {
                // Check if jsPDF is available
                if (typeof window.jspdf === 'undefined') {
                    alert('PDF library not loaded. Please refresh the page and try again.');
                    return;
                }
                
                const { jsPDF } = window.jspdf;
                const doc = new jsPDF();
                
                // Define colors
                const primaryColor = [12, 135, 118]; // #0C8776
                const headerTextColor = [255, 255, 255]; // White
                const textColor = [17, 24, 39]; // Dark gray
                const lightGray = [107, 114, 128]; // #6B7280
                
                // Status colors
                const statusColors = {
                    'Pending': { bg: [254, 243, 199], text: [146, 64, 14] },
                    'Approved': { bg: [209, 250, 229], text: [6, 95, 70] },
                    'Rejected': { bg: [254, 226, 226], text: [153, 27, 27] }
                };
                
                // Header
                doc.setFillColor(...primaryColor);
                doc.rect(0, 0, 210, 40, 'F');
                
                doc.setTextColor(...headerTextColor);
                doc.setFontSize(22);
                doc.setFont('helvetica', 'bold');
                doc.text('Leave History Report', 105, 18, { align: 'center' });
                
                doc.setFontSize(11);
                doc.setFont('helvetica', 'normal');
                doc.text(data.employee_name + ' (' + data.employee_email + ')', 105, 28, { align: 'center' });
                doc.setFontSize(9);
                doc.text('Generated on ' + data.generated_date, 105, 35, { align: 'center' });
                
                // Stats Section
                let startY = 50;
                const statsWidth = 40;
                const statsSpacing = 5;
                const totalStatsWidth = (statsWidth * 4) + (statsSpacing * 3);
                let statsStartX = (210 - totalStatsWidth) / 2;
                
                // Stats boxes
                const statsData = [
                    { label: 'Total', value: data.stats.total, bg: [243, 244, 246] },
                    { label: 'Pending', value: data.stats.pending, bg: [254, 243, 199] },
                    { label: 'Approved', value: data.stats.approved, bg: [209, 250, 229] },
                    { label: 'Rejected', value: data.stats.rejected, bg: [254, 226, 226] }
                ];
                
                statsData.forEach((stat, index) => {
                    const x = statsStartX + (index * (statsWidth + statsSpacing));
                    doc.setFillColor(...stat.bg);
                    doc.roundedRect(x, startY, statsWidth, 22, 3, 3, 'F');
                    
                    doc.setTextColor(...textColor);
                    doc.setFontSize(18);
                    doc.setFont('helvetica', 'bold');
                    doc.text(String(stat.value), x + statsWidth/2, startY + 12, { align: 'center' });
                    
                    doc.setTextColor(...lightGray);
                    doc.setFontSize(8);
                    doc.setFont('helvetica', 'normal');
                    doc.text(stat.label.toUpperCase(), x + statsWidth/2, startY + 19, { align: 'center' });
                });
                
                // Table
                const tableStartY = startY + 35;
                
                // Prepare table data
                const tableHeaders = [['#', 'Leave Type', 'Start Date', 'End Date', 'Days', 'Status', 'Applied On']];
                const tableBody = data.data.map((leave, index) => [
                    index + 1,
                    leave.leave_type,
                    leave.start_date,
                    leave.end_date,
                    leave.days,
                    leave.status,
                    leave.created_at
                ]);
                
                // Use autoTable for the table
                doc.autoTable({
                    head: tableHeaders,
                    body: tableBody,
                    startY: tableStartY,
                    theme: 'grid',
                    headStyles: {
                        fillColor: primaryColor,
                        textColor: headerTextColor,
                        fontStyle: 'bold',
                        fontSize: 9,
                        halign: 'center'
                    },
                    bodyStyles: {
                        fontSize: 9,
                        textColor: textColor,
                        halign: 'center'
                    },
                    columnStyles: {
                        0: { cellWidth: 10 },
                        1: { cellWidth: 28, halign: 'left' },
                        2: { cellWidth: 28 },
                        3: { cellWidth: 28 },
                        4: { cellWidth: 15 },
                        5: { cellWidth: 22 },
                        6: { cellWidth: 28 }
                    },
                    didParseCell: function(hookData) {
                        // Style status column
                        if (hookData.section === 'body' && hookData.column.index === 5) {
                            const status = hookData.cell.raw;
                            const colors = statusColors[status] || statusColors['Pending'];
                            hookData.cell.styles.fillColor = colors.bg;
                            hookData.cell.styles.textColor = colors.text;
                            hookData.cell.styles.fontStyle = 'bold';
                        }
                    },
                    margin: { left: 15, right: 15 }
                });
                
                // Footer
                const pageHeight = doc.internal.pageSize.height;
                doc.setTextColor(...lightGray);
                doc.setFontSize(8);
                doc.text('This is a computer-generated document. No signature is required.', 105, pageHeight - 10, { align: 'center' });
                
                // Download the PDF
                doc.save(data.filename);
            }
        });
        </script>
        <?php
    }

    /**
     * AJAX: Apply for Leave
     */
    public static function handle_ajax_apply_leave() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        $leave_type = sanitize_text_field($_POST['leave_type']);
        $start_date = sanitize_text_field($_POST['start_date']);
        $end_date = sanitize_text_field($_POST['end_date']);
        $reason = sanitize_textarea_field($_POST['reason']);

        // Validate dates
        if (strtotime($end_date) < strtotime($start_date)) {
            wp_send_json_error(['message' => 'End date must be after start date']);
        }

        $result = $wpdb->insert($table_name, [
            'employee_id' => $current_user->ID,
            'leave_type' => $leave_type,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'reason' => $reason,
            'status' => 'pending',
            'created_at' => current_time('mysql')
        ]);

        if ($result) {
            // Create notification for admin
            HRM_Notifications::create_notification(
                1, // Admin user ID
                'leave_request',
                sprintf('%s has applied for %s leave', $current_user->display_name, $leave_type),
                $wpdb->insert_id
            );

            wp_send_json_success(['message' => 'Leave request submitted successfully']);
        } else {
            wp_send_json_error(['message' => 'Error submitting leave request']);
        }
    }

    /**
     * AJAX: Update Leave Status (Admin)
     */
    public static function handle_ajax_update_leave_status() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';

        $leave_id = intval($_POST['leave_id']);
        $status = sanitize_text_field($_POST['status']);
        $admin_remarks = sanitize_textarea_field($_POST['admin_remarks']);

        // Get leave details
        $leave = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d",
            $leave_id
        ));

        if (!$leave) {
            wp_send_json_error(['message' => 'Leave request not found']);
        }

        $result = $wpdb->update(
            $table_name,
            [
                'status' => $status,
                'admin_remarks' => $admin_remarks,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $leave_id]
        );

        if ($result !== false) {
            // Notify employee
            HRM_Notifications::create_notification(
                $leave->employee_id,
                'leave_' . $status,
                sprintf('Your %s leave request has been %s', $leave->leave_type, $status),
                $leave_id
            );

            wp_send_json_success(['message' => 'Leave status updated successfully']);
        } else {
            wp_send_json_error(['message' => 'Error updating leave status']);
        }
    }

    /**
     * AJAX: Cancel Leave
     */
    public static function handle_ajax_cancel_leave() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        $leave_id = intval($_POST['leave_id']);

        // Verify ownership and status
        $leave = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d AND employee_id = %d AND status = 'pending'",
            $leave_id,
            $current_user->ID
        ));

        if (!$leave) {
            wp_send_json_error(['message' => 'Leave request not found or cannot be cancelled']);
        }

        $result = $wpdb->delete($table_name, ['id' => $leave_id]);

        if ($result) {
            wp_send_json_success(['message' => 'Leave request cancelled successfully']);
        } else {
            wp_send_json_error(['message' => 'Error cancelling leave request']);
        }
    }

    /**
     * AJAX: Edit Leave Request
     */
    public static function handle_ajax_edit_leave_request() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        $leave_id = intval($_POST['leave_id']);
        $leave_type = sanitize_text_field($_POST['leave_type']);
        $start_date = sanitize_text_field($_POST['start_date']);
        $end_date = sanitize_text_field($_POST['end_date']);
        $reason = sanitize_textarea_field($_POST['reason']);

        // Verify ownership and status
        $leave = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d AND employee_id = %d AND status = 'pending'",
            $leave_id,
            $current_user->ID
        ));

        if (!$leave) {
            wp_send_json_error(['message' => 'Leave request not found or cannot be edited']);
        }

        // Validate dates
        if (strtotime($end_date) < strtotime($start_date)) {
            wp_send_json_error(['message' => 'End date must be after start date']);
        }

        $result = $wpdb->update(
            $table_name,
            [
                'leave_type' => $leave_type,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'reason' => $reason,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $leave_id]
        );

        if ($result !== false) {
            wp_send_json_success(['message' => 'Leave request updated successfully']);
        } else {
            wp_send_json_error(['message' => 'Error updating leave request']);
        }
    }

    /**
     * AJAX: Get Leave Data
     */
    public static function handle_ajax_get_leave_data() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        $leave_id = intval($_POST['leave_id']);

        $leave = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d AND employee_id = %d",
            $leave_id,
            $current_user->ID
        ));

        if ($leave) {
            wp_send_json_success($leave);
        } else {
            wp_send_json_error(['message' => 'Leave request not found']);
        }
    }

    /**
     * AJAX: Export Leave History as CSV
     */
    public static function handle_ajax_export_leave_csv() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error(['message' => 'Unauthorized access']);
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        // Get leave data for current user only (security)
        $leaves = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE employee_id = %d ORDER BY created_at DESC",
            $current_user->ID
        ));

        if (empty($leaves)) {
            wp_send_json_error(['message' => 'No leave records to export']);
            return;
        }

        // Build CSV content
        $csv_data = [];
        $csv_data[] = ['ID', 'Leave Type', 'Start Date', 'End Date', 'Days', 'Reason', 'Status', 'Admin Remarks', 'Applied On'];

        foreach ($leaves as $leave) {
            $start = new DateTime($leave->start_date);
            $end = new DateTime($leave->end_date);
            $days = $end->diff($start)->days + 1;

            $csv_data[] = [
                $leave->id,
                ucfirst($leave->leave_type),
                date('Y-m-d', strtotime($leave->start_date)),
                date('Y-m-d', strtotime($leave->end_date)),
                $days,
                $leave->reason,
                ucfirst($leave->status),
                $leave->admin_remarks ?: '-',
                date('Y-m-d H:i', strtotime($leave->created_at))
            ];
        }

        // Convert to CSV string
        $csv_string = '';
        foreach ($csv_data as $row) {
            $csv_string .= '"' . implode('","', array_map(function($val) {
                return str_replace('"', '""', $val);
            }, $row)) . '"' . "\n";
        }

        wp_send_json_success([
            'filename' => 'leave_history_' . date('Y-m-d') . '.csv',
            'content' => $csv_string,
            'employee_name' => $current_user->display_name
        ]);
    }

    /**
     * AJAX: Export Leave History as PDF (returns HTML for client-side PDF generation)
     */
    public static function handle_ajax_export_leave_pdf() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error(['message' => 'Unauthorized access']);
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_leave';
        $current_user = wp_get_current_user();

        // Get leave data for current user only (security)
        $leaves = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE employee_id = %d ORDER BY created_at DESC",
            $current_user->ID
        ));

        if (empty($leaves)) {
            wp_send_json_error(['message' => 'No leave records to export']);
            return;
        }

        // Build data for PDF
        $pdf_data = [];
        foreach ($leaves as $leave) {
            $start = new DateTime($leave->start_date);
            $end = new DateTime($leave->end_date);
            $days = $end->diff($start)->days + 1;

            $pdf_data[] = [
                'id' => $leave->id,
                'leave_type' => ucfirst($leave->leave_type),
                'start_date' => date('M d, Y', strtotime($leave->start_date)),
                'end_date' => date('M d, Y', strtotime($leave->end_date)),
                'days' => $days,
                'reason' => $leave->reason,
                'status' => ucfirst($leave->status),
                'admin_remarks' => $leave->admin_remarks ?: '-',
                'created_at' => date('M d, Y', strtotime($leave->created_at))
            ];
        }

        // Calculate statistics
        $stats = [
            'total' => count($leaves),
            'pending' => 0,
            'approved' => 0,
            'rejected' => 0
        ];

        foreach ($leaves as $leave) {
            if (isset($stats[$leave->status])) {
                $stats[$leave->status]++;
            }
        }

        wp_send_json_success([
            'filename' => 'leave_history_' . date('Y-m-d') . '.pdf',
            'data' => $pdf_data,
            'stats' => $stats,
            'employee_name' => $current_user->display_name,
            'employee_email' => $current_user->user_email,
            'generated_date' => date('F d, Y \a\t h:i A')
        ]);
    }
}
